function launcher
% The **launcher** provides a unified entry point to the project’s graphical interfaces. 
% It presents a single selection window from which the user can choose the desired 
% structure editor (standard cell, MV, L-wire, T-wire, custom layout, etc.); 
% upon selection, the corresponding MATLAB GUI is invoked without requiring 
% manual invocation of individual script files.


    % ---- GUI choices
    GUIS = [ ...
        entry('Standard Cell','stdcell_gui', ...
              'GUI per la cella standard singola.') ...
        entry('Standard Cell (CUT-Y Version)','stdcell_gui_cutY_far_flush_with_Ti', ...
              'Variante con taglio Y a filo pellicola di Ti.') ...
        entry('Array N-Phase','stdcell_array_gui', ...
              'Duplica la cella lungo Y: ultime standard, precedenti CUT.') ...
        entry('Majority Voter','stdcell_mv_gui', ...
              'Cross MV con bracci simmetrici e filler stratificati agli angoli.') ...
        entry('Majority Voter (Cross Vacuum Version)','stdcell_mv_crossvac_gui', ...
              'MV con BLOCCO centrale HfO2 + croce di vuoto (spessore bracci = L_vac) e cap Au quadrato.') ...
        entry('L-Wire','stdcell_lwire_gui', ...
              'GUI per struttura a L con 2 bracci perpendicolari per il routing.') ...
        entry('L-Wire (Cross Vacuum Version)','stdcell_lwire_crossvac_gui', ...
              'L-Wire con blocco centrale HfO2 + croce di vuoto.') ...
        entry('T-Wire','stdcell_twire_gui', ...
              'GUI per struttura a T con 3 bracci per il routing dell''informazione.') ...
        entry('T-Wire (Cross Vacuum Version)','stdcell_twire_crossvac_gui', ...
              'T-Wire con blocco centrale HfO2 + croce di vuoto.') ...
        entry('Custom Layout (Canvas + 3D)','custom_layout_gui', ...
              'Canvas 2D + editor 3D per comporre MV/L/T/Bus e generare SDE/SDevice.') ...
    ];

    % ---- window
    ui = uifigure('Name','STD-CELL – Launcher','Color','w','Position',[120 120 520 220]);

    % Layout: title | tends | spacer | bottons
    gl = uigridlayout(ui,[4 1]);
    gl.RowHeight = {40, 60, '1x', 46};
    gl.Padding = [16 14 16 14];

    % Titles
    uilabel(gl,'Text','Choose the structure to edit', ...
        'FontWeight','bold','HorizontalAlignment','left');

    % choices
    labels = {GUIS.label};                 % 1xN cell of char
    dd = uidropdown(gl, ...
        'Items', labels, ...
        'ItemsData', 1:numel(GUIS), ...
        'Value', 1);

    % Spacer
    uilabel(gl,'Text','');

    % buttons
    btnBar = uigridlayout(gl,[1 3]); 
    btnBar.ColumnWidth = {'fit','1x','fit'};
    uibutton(btnBar,'Text','Open', ...
        'BackgroundColor',[0.10 0.55 0.10],'FontColor','w', ...
        'ButtonPushedFcn',@(~,~)launchSelected());
    uilabel(btnBar,'Text',''); % spacer
    uibutton(btnBar,'Text','Close','ButtonPushedFcn',@(~,~)close(ui));

    % ============== nested ==============
    function launchSelected()
        e = GUIS(dd.Value);
        if ~isOnPath(e.func)
            uialert(ui, sprintf(['La funzione "%s" non è stata trovata sul path.\n' ...
                                 'Assicurati che il file sia nella stessa cartella del launcher.'], e.func), ...
                    'Funzione non trovata', 'Icon','warning');
            return;
        end
        try
            feval(e.func);  % launch selected GUI
        catch err
            uialert(ui, sprintf('Errore durante l''apertura di "%s":\n\n%s', e.func, err.message), ...
                   'Errore di esecuzione', 'Icon','error');
        end
    end
end

% ========= helper local =========
function e = entry(label, func, desc)
    if isstring(label), label = char(label); end
    if isstring(func),  func  = char(func);  end
    if isstring(desc),  desc  = char(desc);  end
    e = struct('label',label,'func',func,'desc',desc);
end

function tf = isOnPath(funcname)
    tf = ~isempty(which(funcname)) || exist(funcname,'file')==2;
end
